/**
 * Feed: class responsible for single feed: make request for new items and keeps list of last updated news
 * */
function Feed( url, params ) { 
	
	var params = params || {};
	this.url = url;
	this.title = params.title || '';
	this.color = params.color || '';
	this.itemsLimit = params.itemsLimit || 10;
}

/**
 * setting default values for lastUpdate, unreadNum, items
 * */
Feed.prototype.lastUpdate = new Date(0);
Feed.prototype.unreadNum = 0;
Feed.prototype.items = [];
/**
 * fill items array with new ones
 * @param {Array} of items
 * */
Feed.prototype.setItems = function(items){
	
	items = this.cutExcess(items);
	this.items = [];
	this.unreadNum = 0;
	for( var i=0; i<items.length; i++ ) {
		items[i].setFeed( this );
		this.items.push( items[i] );
		if( !items[i].read ){
			this.unreadNum++;
		}
	}
	this.loaded = true;
};
/**
 * reduce feed items 
 * */
Feed.prototype.cutExcess = function(items){
	items.sort( function(a,b){if(a.pubDate > b.pubDate){return -1;}if(a.pubDate < b.pubDate){return 1;}return 0;} );
	return items.splice(0,this.itemsLimit);
}
/**
 * set lastUpdate time
 * @param {Date} time
 * */
Feed.prototype.setLastUpdate = function(time){
	this.lastUpdate = time;
};
/**
 * called to get or refresh feed items and attributes
 * @method load
 * @param {Function} onSuccess
 * @param {Function} onFailure
 * @return {XMLHttpRequest} sent request
 */

Feed.prototype.load = function(onSuccess, onFailure){
	
	var feed = this;
	var oReq = new O.IO.Req({
        url: this.url,
        proxy: true,
        onSuccess: {
            fn: function(req) {
                try{
                	var respTxt = oReq.getRespText();
                	var items = feed._parseResponse( respTxt );
                	if(!items.length){
            			onFailure.call(feed, 'No items retrieved from RSS.');
        				return;
            		}
                	feed.mergeItems(items); 
            		feed.lastUpdate = new Date(); 
                	feed.loaded = true;
                } catch(e) {
                	onFailure.call(feed, 'Cannot parse items from defined RSS.');
                    return;
                }
                onSuccess.call(feed);
            }
        },
        onFailure: {
            fn: function(req, ioErr) {
                if(ioErr) {
                	onFailure.call(feed,ioErr.msg);
                    return;
                }

                try {
                    var respJSON = JSON.parse(req.responseText);
                }
                catch(e) {
                	onFailure.call(feed,'Cannot parse JSON response.');
                    return;
                }

                onFailure.call(feed, respJSON);
            }
        }});
	return oReq;
};

/**
 * parses items from responseXML
 * @method _parseResponse
 * @private
 * @param {XMLObject} responseXML
 */
Feed.prototype._parseResponse = function( responseTxt ){
	
	var parser = new DOMParser();
	var responseXML = parser.parseFromString( responseTxt,'text/xml' );
    try{
    	var _items = Parser.getItems( responseXML );
    	var _channel = Parser.getChannelData( responseXML );
    	if( _channel ){
    		if( _channel.icon ){
    			this.icon = _channel.icon;
    		}
    		if( _channel.title ){
    			this.title = _channel.title;
    		}
    		if( _channel.description ){
    			this.description = _channel.description;
    		}
    	}
    	
    } catch(err){
    	throw "Problems with data parsing.";
    }
	return _items;
};

/**
 * get feed and retrieves items if it's not loaded yet
 * @method get
 * @param {Function} onReady
 */
Feed.prototype.get = function(onReady){
	
	if(!this.loaded){
		this.load(onReady, onReady);
	}
	else {
		onReady.call(this);
	}
};

/**
 * merge new items with old ones, does not overwrite existing items and removes old items
 * @method mergeItems
 * @param {Array} items
 */
Feed.prototype.mergeItems = function(items){
	var feed = this,
		foundItem,
		item,
		_items = [],
		unreadNum = 0;
	
	for( var i=0; i<items.length; i++ ){
		item = items[i];
		foundItem = feed.find(item);
		if( foundItem ){
			_items.push(foundItem);
		} 
		else {
			item.setFeed(this);
			_items.push(item);
			if( !item.read ){
				unreadNum++;
			}
		}
	}
	this.setItems( _items );
	this.unreadNum = unreadNum;
};
/**
 * finds item with item provided
 * @method find
 * @param {Item} item
 */
Feed.prototype.find = function(item){
	
	for(var i = 0; i < this.items.length; i++){
		if(item.id == this.items[i].id){
			return this.items[i];
		}
	}
	return null;
}
/**
 * get number of unread items
 * */
Feed.prototype.getUnreadNum = function(){
	var num = 0;
	
	for(var i = 0; i < this.items.length; i++){
		if(!this.items[i].read){
			num++;
		}
	}
	this.unreadNum = num;
	return num;
}
/**
 * returns items of feed
 * */
Feed.prototype.getItems = function(){
	return this.items;
};